#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
//libkmod uses syslog constants as priority
#include <syslog.h>

#include "model/kmodule.h"
#include "control/kmodule_fsm.h"
#include "util/logger.h"
#include "util/kmodules_parser.h"
#include "util/generic_conf_parser.h"



struct kmod_ctx *kmodule_ctx=NULL;

typedef struct kmodule_t
{
	struct kmod_module *mod;

	char *mod_params;

	kmodule_state_t state;

	int depmod_cnt;

	kmodule_list_t *waiting_modules;

	kmodule_list_entry_t *mainlist_entry;

} kmodule_t;

static void kmodule_log_kmod_line(void *log_data, int priority, const char *file, int line,
		const char *fn, const char *format, va_list args);

error_code_t kmodule_init(void)
{
	kmodule_ctx=kmod_new(NULL,NULL);
	if (kmodule_ctx==NULL)
		return RESULT_NORESOURCES;

	logger_log_debug("KMODULE -> init done (result: %d).",RESULT_OK);
	kmod_set_log_fn(kmodule_ctx,kmodule_log_kmod_line,NULL);

	//	kmod_set_log_priority(kmodule_ctx,8);
	return RESULT_OK;
}

void kmodule_deinit(void)
{
	if (kmodule_ctx!=NULL)
		kmod_unref(kmodule_ctx);
	logger_log_debug("KMODULE -> deinit done.");
}

error_code_t kmodule_new_from_path(kmodule_t **kmodule, const char *path, const char *params)
{
	error_code_t result;
	struct kmod_module *mod;

	if (kmod_module_new_from_path(kmodule_ctx,path,&mod)!=0)
	{
		(*kmodule)=NULL;
		result=RESULT_MODULE_NOT_FOUND;
	}
	else
		result=kmodule_new_from_mod(kmodule,mod,params);

	// a new reference of mod is checked out in kmodule_new_from_mod
	kmod_module_unref(mod);

	return result;	
}

error_code_t kmodule_new_from_name(kmodule_t **kmodule, const char *name, const char *params)
{
	error_code_t result;
	struct kmod_module *mod=NULL;

	if (kmod_module_new_from_name(kmodule_ctx,name,&mod)!=0)
	{
		(*kmodule)=NULL;
		result=RESULT_MODULE_NOT_FOUND;
	}
	else
		result=kmodule_new_from_mod(kmodule,mod,params);

	// a new reference of mod is checked out in kmodule_new_from_mod
	kmod_module_unref(mod);

	return result;
}

error_code_t kmodule_new_from_mod(kmodule_t **kmodule, struct kmod_module *mod, const char *params)
{
	error_code_t result = RESULT_OK;
	int kmod_state;
	char *parameters = NULL;
	kmodule_t *new_kmodule = NULL;

	kmod_state = kmod_module_get_initstate(mod);
	(*kmodule) = NULL;

	if (kmod_state == KMOD_MODULE_BUILTIN)
		result = RESULT_MODULE_BUILTIN;

	if (kmod_state > 0)
		result = RESULT_MODULE_LOADED;

	if (result == RESULT_OK)
	{
		new_kmodule = malloc(sizeof(kmodule_t));
		(*kmodule) = new_kmodule;
		if (new_kmodule == NULL)
			result = RESULT_NORESOURCES;
	}

	//sorry this is needed here to satisfy lin(t)
	if (new_kmodule == NULL)
		return result;

	if (result == RESULT_OK)
	{

		const char *kmodule_name = kmod_module_get_name(mod);
		result = kmodules_parser_parse_kcmdline(kmodule_name, &parameters);
	}

	if (result == RESULT_OK)
	{
		if (params != NULL)
		{
			char *params_temp = strdup(params);
			if (params_temp != NULL)
			{
				result = generic_conf_parser_add_parameters(&parameters, params_temp);
				free(params_temp);
			}
			else
			{
				result = RESULT_NORESOURCES;
			}
		}
	}

	if (result == RESULT_OK)
	{
		if (parameters != NULL)
		{
			logger_log_debug("KMODULE -> Add parameters: %s", parameters);
			new_kmodule->mod_params = strdup(parameters);
			free(parameters);
			if (new_kmodule->mod_params == NULL)
				result = RESULT_NORESOURCES;
		}
		else
			new_kmodule->mod_params = NULL;
	}

	if (result == RESULT_OK)
	{
		new_kmodule->waiting_modules = kmodule_list_create();
		if (new_kmodule->waiting_modules == NULL)
			result = RESULT_NORESOURCES;
	}

	if (result == RESULT_OK)
	{
		new_kmodule->mod = kmod_module_ref(mod);
		new_kmodule->state = ADDED;
		new_kmodule->depmod_cnt = -1;
		new_kmodule->mainlist_entry = NULL;
	}

	return result;
}

void kmodule_free(kmodule_t *kmodule)
{
	if (kmodule->mod!=NULL)
		kmod_module_unref(kmodule->mod);
	if (kmodule->mod_params!=NULL)
		free(kmodule->mod_params);
	if (kmodule->waiting_modules!=NULL)
		kmodule_list_free(kmodule->waiting_modules);
	free(kmodule);
}

kmodule_state_t kmodule_get_state(kmodule_t *kmodule)
{
	return kmodule->state;
}

void kmodule_set_state_depmod_scheduled(kmodule_t *kmodule, int depmod_cnt)
{
	kmodule->depmod_cnt=depmod_cnt;
	kmodule->state=DEPMOD_SCHEDULED;
	logger_log_debug("KMODULE -> Module %s enters state DEPMOD_SCHEDULED.",kmodule_get_name(kmodule));
}

void kmodule_set_state_ready_to_schedule(kmodule_t *kmodule)
{
	kmodule->state=READY_TO_SCHEDULE;
	logger_log_debug("KMODULE -> Module %s enters state READY_TO_SCHEDULE.",kmodule_get_name(kmodule));
}

void kmodule_set_state_loading(kmodule_t *kmodule)
{
	kmodule->state=LOADING;
	logger_log_debug("KMODULE -> Module %s enters state LOADING.",kmodule_get_name(kmodule));
}

void kmodule_set_mainlist_entry(kmodule_t *kmodule, kmodule_list_entry_t *list_entry_pointer)
{
	kmodule->mainlist_entry=list_entry_pointer;
}

kmodule_list_entry_t *kmodule_get_mainlist_entry(kmodule_t *kmodule)
{
	return kmodule->mainlist_entry;
}

void kmodule_remove_from_main_list(kmodule_t *kmodule)
{
	kmodule_list_entry_t *entry;
	entry=kmodule->mainlist_entry;
	if (entry!=NULL)
		kmodule_list_remove_from_list(entry);
}

bool kmodule_all_depmods_loaded(kmodule_t *kmodule)
{
	return kmodule->depmod_cnt<=0;
}

void kmodule_signal_depmod_loaded(kmodule_t *kmodule)
{
	kmodule->depmod_cnt--;
}

error_code_t kmodule_load(kmodule_t *kmodule)
{   
	int result;
	result=kmod_module_insert_module(kmodule->mod,0,kmodule->mod_params);
	//normally we are not trying to load something if it is there, we check it in advance.
	//But there might occur a case (which actually happened) that someone else (udevd)
	//loads our module while we are waiting for a dependency to be resolved. Actually, loading
	//the module we waited for, triggered udevd to load our module as well. One of both is faster ...
	if (result<0 && result!=-EEXIST)
		return RESULT_MODLOADING_FAILED;
	return RESULT_OK;
}

error_code_t kmodule_add_waiting_module(kmodule_t *kmodule, kmodule_t *waiting_kmodule)
{
	return kmodule_list_queue_module(kmodule->waiting_modules, waiting_kmodule,NULL);
}

void kmodule_signal_done_to_waiting_modules(kmodule_t *kmodule)
{
	kmodule_t *waiting_kmodule;
	while (!kmodule_list_is_empty(kmodule->waiting_modules))
	{
		waiting_kmodule=kmodule_list_dequeue_module(kmodule->waiting_modules);
		kmodule_fsm_signal_depmod_loaded(waiting_kmodule);
	}
}

const char *kmodule_get_name(kmodule_t *kmodule)
{
	return kmod_module_get_name(kmodule->mod);
}

const struct kmod_module *kmodule_get_kmod(kmodule_t *kmodule)
{
	return kmodule->mod;
}

static void kmodule_log_kmod_line(void *log_data, int priority, const char *file, int line,
		const char *fn, const char *format, va_list args)
{
	if (priority==LOG_EMERG || priority==LOG_ALERT || priority==LOG_CRIT || priority==LOG_ERR)
	{
		logger_log_error("Error in function: %s",fn);
		logger_vlog_error(format,args);
	}
	else if (priority==LOG_WARNING || priority==LOG_NOTICE || priority==LOG_INFO)
	{
		logger_log_info("Info in function: %s",fn);
		logger_vlog_info(format,args);
	}
	else
	{
		logger_log_debug("Debug in function: %s",fn);
		logger_vlog_debug(format,args);
	}

	(void)log_data;
	(void)file;
	(void)line;
}
